﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;

namespace MyLittlePhotoshop
{
    public class Filters
    {
        public static Bitmap MapPixels(Bitmap img, Func<Color, Color> f)
        {
            for (int y = 0; y < img.Size.Height; ++y)
                for (int x = 0; x < img.Size.Width; ++x)
                    img.SetPixel(x, y, f(img.GetPixel(x, y)));
            return img;
        }

        private static bool IsValid(int x, int y, Size s)
        {
            return x >= 0 && y >= 0 && x < s.Width && y < s.Height;
        }

        public static Bitmap Convolution(Bitmap img, float[,] mat)
        {
            int maxOffset = mat.GetLength(0)/2;
            Bitmap res = new Bitmap(img);
            for (int y = 0; y < img.Size.Height; ++y)
                for (int x = 0; x < img.Size.Width; ++x)
                {
                    float[] acc = new float[] {0, 0, 0};
                    for (int dy = -maxOffset; dy <= maxOffset; ++dy)
                        for (int dx = -maxOffset; dx <= maxOffset; ++dx)
                            if (IsValid(x + dx, y + dy, img.Size))
                            {
                                Color c = img.GetPixel(x + dx, y + dy);
                                float coef = mat[dx + maxOffset, dy + maxOffset];
                                acc[0] += c.R * coef;
                                acc[1] += c.G * coef;
                                acc[2] += c.B * coef;
                            }
                    for (int i = 0; i < 3; ++i)
                        acc[i] = Math.Max(0, Math.Min(255, acc[i]));
                    res.SetPixel(x, y, Color.FromArgb((int)acc[0], (int)acc[1], (int)acc[2]));
                }
            return res;
        }

        public static Bitmap MirrorH(Bitmap img)
        {
            for (int y = 0; y < img.Size.Height; ++y)
                for (int x = 0; x < img.Size.Width / 2; ++x)
                {
                    Color tmp = img.GetPixel(x, y);
                    img.SetPixel(x, y, img.GetPixel(img.Size.Width - x - 1, y));
                    img.SetPixel(img.Size.Width - x - 1, y, tmp);
                }
            return img;
        }

        public static Bitmap MirrorV(Bitmap img)
        {
            for (int y = 0; y < img.Size.Height / 2; ++y)
                for (int x = 0; x < img.Size.Width; ++x)
                {
                    Color tmp = img.GetPixel(x, y);
                    img.SetPixel(x, y, img.GetPixel(x, img.Size.Height - y - 1));
                    img.SetPixel(x, img.Size.Height - y - 1, tmp);
                }
            return img;
        }

        public static Color Greyscale(Color c)
        {
            int g = (int)(c.R*0.3 + c.G*0.59 + c.B*0.11);
            return Color.FromArgb(g, g, g);
        }

        public static Color Pinkify(Color c)
        {
            int g = Greyscale(c).R;
            return Color.FromArgb((205 * g / 255 + c.R) / 2, (16 * g / 255 + c.G) / 2, (118 * g / 255 + c.B) / 2);
        }

        public static Color Binarize(Color c)
        {
            return Greyscale(c).R >= 128 ? Color.White : Color.Black;
        }

        public static Color Invert(Color c)
        {
            return Color.FromArgb(255 - c.R, 255 - c.G, 255 - c.B);
        }

        public static float[,] AverageMat = new float[,]
                                               {
                                                   {1/9f, 1/9f, 1/9f},
                                                   {1/9f, 1/9f, 1/9f},
                                                   {1/9f, 1/9f, 1/9f}
                                               };

        public static float[,] GaussMat = new float[,]
                                               {
                                                   {1/16f, 2/16f, 1/16f},
                                                   {2/16f, 4/16f, 2/16f},
                                                   {1/16f, 2/16f, 1/16f}
                                               };

        public static float[,] EdgeMat = new float[,]
                                               {
                                                   {0, -1, 0},
                                                   {-1, 0, 1},
                                                   {0, 1, 0}
                                               };
    }
}
